import os
from aiogram import Router, F
from aiogram.types import Message, CallbackQuery
from aiogram.filters import CommandStart, Command
from utils.downloader import get_video_info, download_video, get_platform
from utils.keyboard import quality_keyboard, cancel_keyboard
from utils.database import add_user, log_download
from config import MAX_FILE_SIZE

router = Router()

@router.message(CommandStart())
async def start(message: Message):
    await add_user(message.from_user.id, message.from_user.username or "", message.from_user.first_name)
    await message.answer(
        f"👋 أهلاً وسهلاً <b>{message.from_user.first_name}</b>!\n\n"
        "🤖 <b>أنا بوت تنزيل الفيديوهات الاحترافي</b>\n\n"
        "━━━━━━━━━━━━━━━━━━\n"
        "📌 <b>المنصات المدعومة:</b>\n"
        "▸ يوتيوب  ▸ تيك توك\n"
        "▸ انستجرام  ▸ فيسبوك\n"
        "▸ تويتر/X  ▸ ريديت\n"
        "▸ Vimeo  ▸ Twitch\n"
        "▸ Dailymotion  ▸ Pinterest\n"
        "▸ وأكتر من 1000 موقع تاني!\n"
        "━━━━━━━━━━━━━━━━━━\n\n"
        "💡 <b>ببساطة ابعتلي رابط أي فيديو</b> وأنا هتكفّل بالباقي! 🎬",
        parse_mode="HTML"
    )

@router.message(Command("help"))
async def help_cmd(message: Message):
    await message.answer(
        "📖 <b>دليل الاستخدام</b>\n\n"
        "1️⃣ انسخ رابط أي فيديو\n"
        "2️⃣ ابعته هنا في الشات\n"
        "3️⃣ اختار الجودة المناسبة\n"
        "4️⃣ انتظر وهيتبعتلك الفيديو ✅\n\n"
        "━━━━━━━━━━━━━━━━━━\n"
        "⚙️ <b>الأوامر المتاحة:</b>\n"
        "/start - بدء البوت\n"
        "/help - عرض المساعدة\n"
        "/stats - إحصائياتي",
        parse_mode="HTML"
    )

@router.message(F.text.regexp(r'https?://[^\s]+'))
async def handle_url(message: Message):
    url = message.text.strip()
    wait_msg = await message.answer("🔍 جاري جلب معلومات الفيديو... انتظر لحظة")
    try:
        info = await get_video_info(url)
        platform = get_platform(url)
        title = info.get("title", "غير معروف")[:50]
        duration = info.get("duration", 0)
        mins, secs = divmod(duration, 60)
        hours, mins = divmod(mins, 60)
        if hours > 0:
            duration_str = f"{hours}:{mins:02d}:{secs:02d}"
        else:
            duration_str = f"{mins}:{secs:02d}"
        uploader = info.get("uploader", info.get("channel", "غير معروف"))
        text = (
            f"✅ <b>تم العثور على الفيديو!</b>\n\n"
            f"🌐 <b>المنصة:</b> {platform}\n"
            f"🎬 <b>العنوان:</b> {title}\n"
            f"👤 <b>القناة:</b> {uploader}\n"
            f"⏱ <b>المدة:</b> {duration_str}\n\n"
            f"📥 <b>اختر جودة التنزيل:</b>"
        )
        await wait_msg.edit_text(text, parse_mode="HTML", reply_markup=quality_keyboard(url))
    except Exception as e:
        await wait_msg.edit_text(
            "❌ <b>عذراً، لم أتمكن من معالجة هذا الرابط</b>\n\n"
            "🔸 تأكد أن الرابط صحيح\n"
            "🔸 تأكد أن الفيديو غير محمي أو مقيّد\n"
            "🔸 حاول مرة أخرى لاحقاً",
            parse_mode="HTML"
        )

@router.message(F.text & ~F.text.startswith("/"))
async def unknown_message(message: Message):
    await message.answer(
        "🤔 لم أفهم طلبك!\n\n"
        "📌 <b>أرسل لي رابط فيديو</b> من أي منصة وسأقوم بتنزيله فوراً",
        parse_mode="HTML"
    )

@router.callback_query(F.data.startswith("dl:"))
async def handle_download(callback: CallbackQuery):
    await callback.answer("⏳ جاري التنزيل...")
    parts = callback.data.split(":", 2)
    quality = parts[1]
    url = parts[2]
    audio_only = quality == "audio"
    quality_label = "MP3 🎵" if audio_only else f"{quality}p 📹"
    progress_msg = await callback.message.edit_text(
        f"⬇️ <b>جاري تنزيل الفيديو</b>\n\n"
        f"🎯 الجودة المختارة: <b>{quality_label}</b>\n"
        f"⏳ يرجى الانتظار...",
        parse_mode="HTML",
        reply_markup=cancel_keyboard()
    )
    filename = None
    try:
        result = await download_video(url, quality, audio_only)
        filename = result["filename"]
        filesize = result["filesize"]
        if filesize > MAX_FILE_SIZE:
            await progress_msg.edit_text(
                f"❌ <b>الفيديو كبير جداً!</b>\n\n"
                f"📦 الحجم: <b>{filesize:.1f} MB</b>\n"
                f"⚠️ الحد الأقصى: <b>{MAX_FILE_SIZE} MB</b>\n\n"
                f"جرّب اختيار جودة أقل",
                parse_mode="HTML"
            )
            os.remove(filename)
            return
        await progress_msg.edit_text(
            f"📤 <b>جاري رفع الملف...</b>\n\n"
            f"📦 الحجم: <b>{filesize:.1f} MB</b>\n"
            f"⏳ انتظر قليلاً...",
            parse_mode="HTML"
        )
        if audio_only:
            with open(filename, "rb") as f:
                await callback.message.answer_audio(
                    f, title=result["title"],
                    caption=(
                        f"🎵 <b>{result['title']}</b>\n\n"
                        f"📦 الحجم: {filesize:.1f} MB\n"
                        f"🤖 @Downloadultra1_bot"
                    ),
                    parse_mode="HTML"
                )
        else:
            with open(filename, "rb") as f:
                await callback.message.answer_video(
                    f,
                    caption=(
                        f"🎬 <b>{result['title']}</b>\n\n"
                        f"🎯 الجودة: {quality_label}\n"
                        f"📦 الحجم: {filesize:.1f} MB\n"
                        f"🤖 @Downloadultra1_bot"
                    ),
                    parse_mode="HTML",
                    supports_streaming=True
                )
        await progress_msg.edit_text(
            "✅ <b>تم التنزيل بنجاح!</b>\n\nأرسل لي رابط آخر في أي وقت 😊",
            parse_mode="HTML"
        )
        await log_download(callback.from_user.id, url, get_platform(url), quality_label, filesize, "success")
    except Exception as e:
        await progress_msg.edit_text(
            f"❌ <b>فشل التنزيل!</b>\n\n"
            f"🔸 السبب: {str(e)[:150]}\n\n"
            f"حاول مرة أخرى أو جرّب جودة مختلفة",
            parse_mode="HTML"
        )
        await log_download(callback.from_user.id, url, get_platform(url), quality, 0, "failed")
    finally:
        if filename and os.path.exists(filename):
            os.remove(filename)

@router.callback_query(F.data == "cancel")
async def cancel(callback: CallbackQuery):
    await callback.answer("تم الإلغاء ✅")
    await callback.message.edit_text("❌ <b>تم إلغاء التنزيل</b>", parse_mode="HTML")
